document.addEventListener('DOMContentLoaded', function () {
    const generateBtn = document.getElementById('generateTokenBtn');
    console.log('Generate button found:', generateBtn);

    if (generateBtn) {
        generateBtn.addEventListener('click', async function () {
            const examId = this.dataset.examId;
            const examLinkInput = document.getElementById('examLink');
            const tokenInfo = document.getElementById('tokenInfo');

            console.log('Button clicked!');
            console.log('Exam ID:', examId);
            console.log('Input element:', examLinkInput);
            console.log('Token info element:', tokenInfo);

            if (!examId) {
                alert('Error: Exam ID tidak ditemukan');
                return;
            }

            try {
                // Disable button during request
                this.disabled = true;
                this.textContent = 'Generating...';

                // Construct API URL
                const currentPath = window.location.pathname;
                let apiUrl;
                
                if (currentPath.includes('/teacher/')) {
                    apiUrl = `../api/generate_token.php?exam_id=${examId}&_=${Date.now()}`;
                } else {
                    apiUrl = `api/generate_token.php?exam_id=${examId}&_=${Date.now()}`;
                }
                
                console.log('Making request to:', apiUrl);
                console.log('Full URL:', new URL(apiUrl, window.location.href).href);
                
                const response = await fetch(apiUrl, {
                    method: 'GET',
                    headers: {
                        'Accept': 'application/json',
                        'Content-Type': 'application/json'
                    },
                    credentials: 'same-origin'
                });

                console.log('Response received:');
                console.log('- Status:', response.status);
                console.log('- OK:', response.ok);
                console.log('- Headers:', Object.fromEntries(response.headers.entries()));

                if (!response.ok) {
                    const errorText = await response.text();
                    console.error('Error response text:', errorText);
                    throw new Error(`HTTP ${response.status}: ${errorText}`);
                }

                // Parse response
                const responseText = await response.text();
                console.log('Raw response text:', responseText);

                let data;
                try {
                    data = JSON.parse(responseText);
                } catch (parseError) {
                    console.error('JSON parse error:', parseError);
                    throw new Error('Invalid JSON response: ' + responseText);
                }

                console.log('Parsed data:', data);

                if (data.success) {
                    console.log('Success! Processing response...');
                    
                    const link = data.exam_url;
                    console.log('Generated link:', link);
                    
                    // Update input field
                    if (examLinkInput) {
                        examLinkInput.value = link;
                        console.log('Input field updated');
                    } else {
                        console.warn('examLink input not found!');
                    }

                    // Update token info
                    if (tokenInfo) {
                        const validUntil = new Date(data.valid_until);
                        console.log('Valid until:', validUntil);
                        
                        tokenInfo.innerHTML = `
                            <strong>Token berhasil dibuat!</strong><br>
                            Berlaku hingga: ${validUntil.toLocaleString('id-ID')}
                        `;
                        tokenInfo.style.color = 'green';
                        tokenInfo.className = 'alert alert-success';
                        console.log('Token info updated');
                    } else {
                        console.warn('tokenInfo element not found!');
                    }

                    // Optional: open link in new tab
                    if (confirm('Buka link ujian di tab baru?')) {
                        window.open(link, '_blank');
                    }
                } else {
                    console.error('API returned success=false:', data);
                    throw new Error(data.message || 'Gagal membuat token');
                }
            } catch (error) {
                console.error('Catch block - Error details:');
                console.error('- Message:', error.message);
                console.error('- Stack:', error.stack);
                
                if (tokenInfo) {
                    tokenInfo.innerHTML = `<strong style="color:red">Error: ${error.message}</strong>`;
                    tokenInfo.className = 'alert alert-danger';
                } else {
                    console.error('Cannot display error - tokenInfo element not found');
                    alert('Error: ' + error.message);
                }
            } finally {
                // Re-enable button
                this.disabled = false;
                this.textContent = 'Generate Link Ujian';
                console.log('Button re-enabled');
            }
        });
    } else {
        console.error('Generate button not found! Check element ID: generateTokenBtn');
    }
});